bl_info = {
    "name": "Custom Marker",
    "author": "Assistant",
    "version": (7, 3, 0),
    "blender": (3, 0, 0),
    "location": "Timeline/Dopesheet/NLA > Sidebar > Custom Markers",
    "description": "Custom marker system with advanced filter rules and group management",
    "category": "Animation",
}

import bpy
import blf
import gpu
from gpu_extras.batch import batch_for_shader
from mathutils import Vector
from bpy.types import Panel, Operator, PropertyGroup
from bpy.props import StringProperty, BoolProperty, CollectionProperty, IntProperty, FloatVectorProperty
from bpy.app.handlers import persistent

# ================================
# Utils
# ================================

def update_all_areas():
    """すべてのエリアを更新"""
    wm = bpy.context.window_manager if bpy.context else None
    if not wm:
        return
    for window in wm.windows:
        for area in window.screen.areas:
            area.tag_redraw()

def ensure_filters(scene):
    """シーンに初期フィルター(A/B)が無ければ作る（register 時には呼ばない）"""
    if not hasattr(scene, "marker_filters"):
        return
    if len(scene.marker_filters) == 0:
        f1 = scene.marker_filters.add()
        f1.internal_group = "A"
        f1.filter_text = ""
        f1.color = (1.0, 0.3, 0.3, 1.0)
        f2 = scene.marker_filters.add()
        f2.internal_group = "B"
        f2.filter_text = ""
        f2.color = (0.3, 0.3, 1.0, 1.0)

def focus_frame_in_editors(frame):
    """指定フレームが画面内に表示されるようにビューを調整"""
    context = bpy.context
    if not context:
        return
    
    # 現在のウィンドウマネージャーから全ウィンドウを処理
    wm = context.window_manager
    if not wm:
        return
    
    # 対象フレームをカレントフレームに設定
    context.scene.frame_set(frame)
    
    for window in wm.windows:
        for area in window.screen.areas:
            # Timeline, Dopesheet, Graph Editor, NLA Editorに対して処理
            if area.type in {'DOPESHEET_EDITOR', 'GRAPH_EDITOR', 'NLA_EDITOR'}:
                space = area.spaces.active
                
                # 適切なリージョンを探す
                for region in area.regions:
                    if region.type == 'WINDOW':
                        # Override contextを作成
                        override = context.copy()
                        override['area'] = area
                        override['region'] = region
                        override['space_data'] = space
                        override['window'] = window
                        override['screen'] = window.screen
                        
                        try:
                            # エディタのタイプによって適切なビュー操作を実行
                            if area.type == 'DOPESHEET_EDITOR':
                                # ドープシートエディタの場合（Timeline含む）
                                if hasattr(space, 'mode'):
                                    with context.temp_override(**override):
                                        try:
                                            if space.mode == 'TIMELINE':
                                                # Timelineモードの場合 - action.view_frameを使用
                                                bpy.ops.action.view_frame()
                                            else:
                                                # Action Editor、その他のドープシートモードの場合
                                                bpy.ops.action.view_frame()
                                        except:
                                            # view_frameが失敗した場合は何もしない
                                            print(f"view_frame failed for {space.mode}")
                                                
                            elif area.type == 'GRAPH_EDITOR':
                                # Graph Editorの場合
                                with context.temp_override(**override):
                                    try:
                                        bpy.ops.graph.view_frame()
                                    except:
                                        print("graph.view_frame failed")
                                        
                            elif area.type == 'NLA_EDITOR':
                                # NLA Editorの場合
                                with context.temp_override(**override):
                                    try:
                                        bpy.ops.nla.view_frame()
                                    except:
                                        # NLA Editorでview_frameが存在しない場合
                                        # 現在のビューを維持（何もしない）
                                        print("nla.view_frame not available")
                            
                        except Exception as e:
                            # 全ての方法が失敗した場合のフォールバック
                            print(f"Focus frame error in {area.type}: {e}")
                            
                        area.tag_redraw()
                        break

# フレーム番号入力用のコールバック関数
def jump_to_input_frame(self, context):
    """入力されたフレーム番号にジャンプする"""
    if self.frame_input > 0:
        context.scene.frame_set(self.frame_input)
        focus_frame_in_editors(self.frame_input)
        update_all_areas()

def get_selected_keyframes(context):
    """選択中のキーフレームのフレーム番号リストを取得"""
    keyframes = []
    
    # アクティブオブジェクトのアニメーションデータを確認
    obj = context.active_object
    if obj and obj.animation_data and obj.animation_data.action:
        action = obj.animation_data.action
        for fcurve in action.fcurves:
            for keyframe in fcurve.keyframe_points:
                if keyframe.select_control_point:
                    frame = int(keyframe.co[0])
                    if frame not in keyframes:
                        keyframes.append(frame)
    
    return sorted(keyframes)

def move_selected_keyframes(context, offset):
    """選択中のキーフレームを指定オフセット分移動"""
    obj = context.active_object
    if not obj or not obj.animation_data or not obj.animation_data.action:
        return False
    
    action = obj.animation_data.action
    
    # 選択中のキーフレームのフレーム番号を取得
    selected_frames = get_selected_keyframes(context)
    if not selected_frames:
        return False
    
    # 先頭のフレーム番号を使用
    source_frame = selected_frames[0]
    target_frame = source_frame + offset
    
    # すべてのFカーブで、source_frameにあるキーフレームを収集
    keyframes_data = []
    
    for fcurve in action.fcurves:
        source_keyframe = None
        source_index = None
        
        # source_frameにあるキーフレームを探す（選択状態に関わらず）
        for i, keyframe in enumerate(fcurve.keyframe_points):
            if abs(keyframe.co[0] - source_frame) < 0.5:
                source_keyframe = keyframe
                source_index = i
                break
        
        if source_keyframe:
            # キーフレーム情報を保存
            keyframes_data.append({
                'fcurve': fcurve,
                'source_index': source_index,
                'value': source_keyframe.co[1],
                'handle_left': source_keyframe.handle_left_type,
                'handle_right': source_keyframe.handle_right_type,
                'interpolation': source_keyframe.interpolation
            })
    
    if not keyframes_data:
        return False
    
    # 古いキーフレームを削除
    for data in keyframes_data:
        fcurve = data['fcurve']
        source_index = data['source_index']
        fcurve.keyframe_points.remove(fcurve.keyframe_points[source_index])
    
    # 新しい位置に作成
    for data in keyframes_data:
        fcurve = data['fcurve']
        new_kf = fcurve.keyframe_points.insert(target_frame, data['value'])
        new_kf.handle_left_type = data['handle_left']
        new_kf.handle_right_type = data['handle_right']
        new_kf.interpolation = data['interpolation']
        new_kf.select_control_point = True
        
        fcurve.update()
    
    return True

def get_selected_keyframe_frame(self):
    """選択中のキーフレームのフレーム番号を取得"""
    context = bpy.context
    selected_frames = get_selected_keyframes(context)
    if selected_frames:
        return selected_frames[0]
    return 1

def set_selected_keyframe_frame(self, value):
    """選択中のキーフレームのフレーム番号を設定"""
    context = bpy.context
    
    # アクティブオブジェクトとアニメーションデータの確認
    if not context.active_object or not context.active_object.animation_data:
        return
    
    # 最新の選択状態を強制的に取得
    selected_frames = get_selected_keyframes(context)
    
    if not selected_frames:
        return
    
    # 先頭フレームとの差分を計算
    first_frame = selected_frames[0]
    offset = value - first_frame
    
    if offset != 0:
        # キーフレームを移動
        if move_selected_keyframes(context, offset):
            # ビューを更新
            update_all_areas()

def copy_selected_keyframe_to_frames(context, target_frames):
    """選択中のキーフレーム（先頭）を指定フレームに複製"""
    obj = context.active_object
    if not obj or not obj.animation_data or not obj.animation_data.action:
        return False
    
    action = obj.animation_data.action
    selected_frames = get_selected_keyframes(context)
    
    if not selected_frames:
        return False
    
    # 先頭のキーフレームを参照
    source_frame = selected_frames[0]
    
    # 各Fカーブで先頭フレームのキーフレームを探して複製
    for fcurve in action.fcurves:
        source_keyframe = None
        for keyframe in fcurve.keyframe_points:
            if int(keyframe.co[0]) == source_frame and keyframe.select_control_point:
                source_keyframe = keyframe
                break
        
        if source_keyframe:
            # 対象フレームに複製
            for target_frame in target_frames:
                # 既存のキーフレームを確認
                existing = None
                for kf in fcurve.keyframe_points:
                    if int(kf.co[0]) == target_frame:
                        existing = kf
                        break
                
                if existing:
                    # 既存のキーフレームを更新
                    existing.co[1] = source_keyframe.co[1]
                    existing.handle_left_type = source_keyframe.handle_left_type
                    existing.handle_right_type = source_keyframe.handle_right_type
                else:
                    # 新規キーフレームを追加
                    new_kf = fcurve.keyframe_points.insert(target_frame, source_keyframe.co[1])
                    new_kf.handle_left_type = source_keyframe.handle_left_type
                    new_kf.handle_right_type = source_keyframe.handle_right_type
            
            fcurve.update()
    
    return True

def copy_and_move_selected_keyframe_to_frames(context, target_frames):
    """選択中のキーフレーム（先頭）を指定フレームに複製し、元のキーフレームを削除
    ただし、元のフレームがtarget_framesに含まれる場合は削除しない"""
    obj = context.active_object
    if not obj or not obj.animation_data or not obj.animation_data.action:
        return False
    
    action = obj.animation_data.action
    selected_frames = get_selected_keyframes(context)
    
    if not selected_frames:
        return False
    
    # 先頭のキーフレームのフレーム番号を参照
    source_frame = selected_frames[0]
    
    # 元のフレームがターゲットに含まれるかチェック
    should_delete_source = source_frame not in target_frames
    
    # すべてのFカーブで、source_frameにあるキーフレーム情報を収集（選択状態に関わらず）
    fcurve_data_list = []
    
    for fcurve in action.fcurves:
        source_keyframe = None
        source_index = None
        
        # source_frameにあるキーフレームを探す（選択状態に関わらず）
        for i, keyframe in enumerate(fcurve.keyframe_points):
            if abs(keyframe.co[0] - source_frame) < 0.5:
                source_keyframe = keyframe
                source_index = i
                break
        
        if source_keyframe:
            # キーフレーム情報を保存
            keyframe_data = {
                'fcurve': fcurve,
                'source_index': source_index,
                'value': source_keyframe.co[1],
                'handle_left': source_keyframe.handle_left_type,
                'handle_right': source_keyframe.handle_right_type,
                'interpolation': source_keyframe.interpolation
            }
            fcurve_data_list.append(keyframe_data)
    
    if not fcurve_data_list:
        return False
    
    # 元のキーフレームを削除（ターゲットに含まれない場合のみ）
    if should_delete_source:
        for data in fcurve_data_list:
            fcurve = data['fcurve']
            source_index = data['source_index']
            fcurve.keyframe_points.remove(fcurve.keyframe_points[source_index])
    
    # 対象フレームに複製
    for data in fcurve_data_list:
        fcurve = data['fcurve']
        value = data['value']
        handle_left = data['handle_left']
        handle_right = data['handle_right']
        interpolation = data['interpolation']
        
        for target_frame in target_frames:
            # 既存のキーフレームを確認
            existing = None
            for kf in fcurve.keyframe_points:
                if abs(kf.co[0] - target_frame) < 0.5:
                    existing = kf
                    break
            
            if existing:
                # 既存のキーフレームを更新
                existing.co[1] = value
                existing.handle_left_type = handle_left
                existing.handle_right_type = handle_right
                existing.interpolation = interpolation
            else:
                # 新規キーフレームを追加
                new_kf = fcurve.keyframe_points.insert(target_frame, value)
                new_kf.handle_left_type = handle_left
                new_kf.handle_right_type = handle_right
                new_kf.interpolation = interpolation
        
        fcurve.update()
    
    return True

def delete_selected_keyframes(context):
    """選択中のキーフレームを削除"""
    obj = context.active_object
    if not obj or not obj.animation_data or not obj.animation_data.action:
        return False
    
    action = obj.animation_data.action
    
    for fcurve in action.fcurves:
        # 削除するキーフレームのインデックスを収集（逆順で削除）
        to_delete = []
        for i, keyframe in enumerate(fcurve.keyframe_points):
            if keyframe.select_control_point:
                to_delete.append(i)
        
        # 逆順で削除
        for i in reversed(to_delete):
            fcurve.keyframe_points.remove(fcurve.keyframe_points[i])
        
        if to_delete:
            fcurve.update()
    
    return True

# ================================
# Properties
# ================================

class CustomMarker(PropertyGroup):
    """カスタムマーカーの情報を保持"""
    name: StringProperty(
        name="Name",
        description="Marker name",
        default="Marker"
    )
    frame: IntProperty(
        name="Frame",
        description="Frame number",
        default=1,
        min=1
    )
    color: FloatVectorProperty(
        name="Color",
        description="Marker color",
        subtype='COLOR',
        size=4,
        default=(1.0, 0.8, 0.2, 1.0),
        min=0.0,
        max=1.0
    )
    group: StringProperty(
        name="Group",
        description="Marker group for filtering",
        default=""
    )

class MarkerFilter(PropertyGroup):
    """フィルター設定を保持するプロパティグループ"""
    filter_text: StringProperty(
        name="Filter",
        description="Custom name for markers",
        default=""
    )
    is_visible: BoolProperty(
        name="Visible",
        description="Toggle visibility for filtered markers",
        default=True,
        update=lambda self, context: update_all_areas()
    )
    color: FloatVectorProperty(
        name="Color",
        description="Group color",
        subtype='COLOR',
        size=4,
        default=(1.0, 0.8, 0.2, 1.0),
        min=0.0,
        max=1.0
    )
    internal_group: StringProperty(
        name="Internal Group",
        description="Internal group identifier (A, B, C...)",
        default=""
    )

class FrameRangePreset(PropertyGroup):
    """フレーム範囲プリセットを保持するプロパティグループ"""
    start_frame: IntProperty(
        name="Start Frame",
        description="Start frame of the range",
        default=1,
        min=0
    )
    end_frame: IntProperty(
        name="End Frame",
        description="End frame of the range",
        default=250,
        min=1
    )
    
    def get_name(self):
        """範囲名を取得（例：1-250）"""
        return f"{self.start_frame}-{self.end_frame}"

# ================================
# Core helpers
# ================================

def get_visible_markers(scene):
    """表示するマーカーをフィルタリングして返す"""
    ensure_filters(scene)

    visible_markers = []
    visible_groups = [f.internal_group for f in scene.marker_filters if f.is_visible]

    for marker in scene.custom_markers:
        if marker.group in visible_groups:
            visible_markers.append(marker)
    return visible_markers

# ================================
# Operators - Frame Range Presets
# ================================

class FRAME_RANGE_OT_add(Operator):
    """フレーム範囲プリセットを追加"""
    bl_idname = "frame_range.add_preset"
    bl_label = "Register Range"
    bl_description = "Register current frame range as a preset"

    def execute(self, context):
        scene = context.scene
        
        start = scene.frame_range_preset_start
        end = scene.frame_range_preset_end
        
        # 開始フレームが終了フレームより大きい場合はエラー
        if start > end:
            self.report({'WARNING'}, "Start frame must be less than or equal to end frame")
            return {'CANCELLED'}
        
        # 新しいプリセットを追加
        preset = scene.frame_range_presets.add()
        preset.start_frame = start
        preset.end_frame = end
        
        self.report({'INFO'}, f"Added preset '{preset.get_name()}'")
        return {'FINISHED'}

class FRAME_RANGE_OT_apply(Operator):
    """フレーム範囲プリセットを適用"""
    bl_idname = "frame_range.apply_preset"
    bl_label = "Apply Frame Range"
    bl_description = "Apply this frame range preset"

    preset_index: IntProperty()

    def execute(self, context):
        scene = context.scene
        
        if 0 <= self.preset_index < len(scene.frame_range_presets):
            preset = scene.frame_range_presets[self.preset_index]
            scene.frame_start = preset.start_frame
            scene.frame_end = preset.end_frame
            
            # キーフレーム操作
            selected_frames = get_selected_keyframes(context)
            
            if selected_frames:
                # オプション1: 開始と終了に複製後、元を削除（被る場合は残す）
                if scene.copy_keyframe_to_range_start_end:
                    target_frames = [preset.start_frame, preset.end_frame]
                    if copy_and_move_selected_keyframe_to_frames(context, target_frames):
                        self.report({'INFO'}, f"Copied keyframe to {preset.start_frame} and {preset.end_frame}")
                
                # オプション2: 開始と終了+1Fに複製後、元を削除（被る場合は残す）
                if scene.copy_keyframe_to_range_start_end_plus1:
                    target_frames = [preset.start_frame, preset.end_frame + 1]
                    if copy_and_move_selected_keyframe_to_frames(context, target_frames):
                        self.report({'INFO'}, f"Copied keyframe to {preset.start_frame} and {preset.end_frame + 1}")
            
            # 開始フレームにフォーカス
            focus_frame_in_editors(preset.start_frame)
            
            self.report({'INFO'}, f"Applied '{preset.get_name()}': {preset.start_frame}-{preset.end_frame}")
        else:
            self.report({'WARNING'}, "Invalid preset index")
            return {'CANCELLED'}
        
        update_all_areas()
        return {'FINISHED'}

class FRAME_RANGE_OT_remove(Operator):
    """フレーム範囲プリセットを削除"""
    bl_idname = "frame_range.remove_preset"
    bl_label = "Remove Frame Range Preset"
    bl_description = "Remove this frame range preset"

    preset_index: IntProperty()

    def invoke(self, context, event):
        return context.window_manager.invoke_confirm(self, event)

    def execute(self, context):
        scene = context.scene
        
        if 0 <= self.preset_index < len(scene.frame_range_presets):
            preset_name = scene.frame_range_presets[self.preset_index].get_name()
            scene.frame_range_presets.remove(self.preset_index)
            self.report({'INFO'}, f"Removed preset '{preset_name}'")
        
        return {'FINISHED'}

class FRAME_RANGE_OT_clear_all(Operator):
    """すべてのフレーム範囲プリセットをクリア"""
    bl_idname = "frame_range.clear_all_presets"
    bl_label = "Clear All Presets"
    bl_description = "Remove all frame range presets"

    def invoke(self, context, event):
        return context.window_manager.invoke_confirm(self, event)

    def execute(self, context):
        context.scene.frame_range_presets.clear()
        self.report({'INFO'}, "Cleared all frame range presets")
        return {'FINISHED'}

# ================================
# Operators - Custom Markers
# ================================

class CUSTOM_MARKER_OT_add(Operator):
    """カスタムマーカーを追加"""
    bl_idname = "custom_marker.add"
    bl_label = "Add Marker"
    bl_description = "Add a custom marker at current frame for this filter"

    filter_index: IntProperty()

    def execute(self, context):
        scene = context.scene
        ensure_filters(scene)

        # 指定されたフィルターを取得
        if 0 <= self.filter_index < len(scene.marker_filters):
            filter_item = scene.marker_filters[self.filter_index]
        else:
            self.report({'WARNING'}, "Invalid filter index")
            return {'CANCELLED'}

        # フィルターテキストが空の場合は現在のフレーム番号を名前にする
        marker_name = filter_item.filter_text or str(scene.frame_current)

        # 同名回避
        existing_names = [m.name for m in scene.custom_markers if m.group == filter_item.internal_group]
        if marker_name in existing_names:
            counter = 2
            while f"{marker_name}_{counter}" in existing_names:
                counter += 1
            marker_name = f"{marker_name}_{counter}"

        marker = scene.custom_markers.add()
        marker.frame = scene.frame_current
        marker.group = filter_item.internal_group
        marker.name = marker_name
        marker.color = filter_item.color

        update_all_areas()
        self.report({'INFO'}, f"Added marker '{marker.name}' (Group {filter_item.internal_group}) at frame {marker.frame}")
        return {'FINISHED'}

class CUSTOM_MARKER_OT_remove(Operator):
    """選択したカスタムマーカーを削除"""
    bl_idname = "custom_marker.remove"
    bl_label = "Remove Custom Marker"
    bl_description = "Remove selected custom marker"

    marker_index: IntProperty()

    def execute(self, context):
        scene = context.scene
        if 0 <= self.marker_index < len(scene.custom_markers):
            marker_name = scene.custom_markers[self.marker_index].name
            scene.custom_markers.remove(self.marker_index)
            update_all_areas()
            self.report({'INFO'}, f"Removed marker '{marker_name}'")
        return {'FINISHED'}

class CUSTOM_MARKER_OT_jump_to(Operator):
    """マーカーのフレームにジャンプ"""
    bl_idname = "custom_marker.jump_to"
    bl_label = "Jump to Marker"
    bl_description = "Jump to marker frame"

    frame: IntProperty()

    def execute(self, context):
        context.scene.frame_set(self.frame)
        # フレームにフォーカス
        focus_frame_in_editors(self.frame)
        update_all_areas()
        return {'FINISHED'}

class CUSTOM_MARKER_OT_jump_to_end(Operator):
    """末端のフレームにジャンプ"""
    bl_idname = "custom_marker.jump_to_end"
    bl_label = "Jump to End"
    bl_description = "Jump to first or last frame"

    direction: StringProperty(default="LAST")

    def execute(self, context):
        scene = context.scene
        if self.direction == "FIRST":
            scene.frame_set(scene.frame_start)
            focus_frame_in_editors(scene.frame_start)
            self.report({'INFO'}, f"Jumped to first frame {scene.frame_start}")
        else:
            scene.frame_set(scene.frame_end)
            focus_frame_in_editors(scene.frame_end)
            self.report({'INFO'}, f"Jumped to last frame {scene.frame_end}")
        update_all_areas()
        return {'FINISHED'}

class CUSTOM_MARKER_OT_jump_to_standard(Operator):
    """標準マーカーにジャンプ"""
    bl_idname = "custom_marker.jump_to_standard"
    bl_label = "Jump to Standard Marker"
    bl_description = "Jump to previous/next standard marker"

    direction: StringProperty(default="NEXT")

    def execute(self, context):
        scene = context.scene
        current_frame = scene.frame_current
        markers = sorted(scene.timeline_markers, key=lambda m: m.frame)

        if not markers:
            self.report({'WARNING'}, "No standard markers in scene")
            return {'CANCELLED'}

        target_frame = None
        target_name = None

        if self.direction == "NEXT":
            for marker in markers:
                if marker.frame > current_frame:
                    target_frame = marker.frame
                    target_name = marker.name
                    break
            if target_frame is None:
                target_frame = markers[0].frame
                target_name = markers[0].name
        else:
            for marker in reversed(markers):
                if marker.frame < current_frame:
                    target_frame = marker.frame
                    target_name = marker.name
                    break
            if target_frame is None:
                target_frame = markers[-1].frame
                target_name = markers[-1].name

        scene.frame_set(target_frame)
        focus_frame_in_editors(target_frame)
        update_all_areas()
        self.report({'INFO'}, f"Jumped to marker '{target_name}' at frame {target_frame}")
        return {'FINISHED'}

class CUSTOM_MARKER_OT_jump_to_custom(Operator):
    """カスタムマーカーにジャンプ"""
    bl_idname = "custom_marker.jump_to_custom"
    bl_label = "Jump to Custom Marker"
    bl_description = "Jump to previous/next custom marker"

    direction: StringProperty(default="NEXT")

    def execute(self, context):
        scene = context.scene
        ensure_filters(scene)

        current_frame = scene.frame_current
        visible_markers = get_visible_markers(scene)
        if not visible_markers:
            self.report({'WARNING'}, "No visible custom markers")
            return {'CANCELLED'}

        markers = sorted(visible_markers, key=lambda m: m.frame)
        target_frame = None
        target_name = None

        if self.direction == "NEXT":
            for marker in markers:
                if marker.frame > current_frame:
                    target_frame = marker.frame
                    target_name = marker.name
                    break
            if target_frame is None:
                target_frame = markers[0].frame
                target_name = markers[0].name
        else:
            for marker in reversed(markers):
                if marker.frame < current_frame:
                    target_frame = marker.frame
                    target_name = marker.name
                    break
            if target_frame is None:
                target_frame = markers[-1].frame
                target_name = markers[-1].name

        scene.frame_set(target_frame)
        focus_frame_in_editors(target_frame)
        update_all_areas()
        self.report({'INFO'}, f"Jumped to custom marker '{target_name}' at frame {target_frame}")
        return {'FINISHED'}

class CUSTOM_MARKER_OT_clear_all(Operator):
    """すべてのカスタムマーカーをクリア"""
    bl_idname = "custom_marker.clear_all"
    bl_label = "Clear All Markers"
    bl_description = "Remove all custom markers"

    def invoke(self, context, event):
        return context.window_manager.invoke_confirm(self, event)

    def execute(self, context):
        context.scene.custom_markers.clear()
        update_all_areas()
        self.report({'INFO'}, "Cleared all custom markers")
        return {'FINISHED'}

class MARKER_OT_clear_filters(Operator):
    """フィルターを初期状態に戻す"""
    bl_idname = "marker.clear_filters"
    bl_label = "Reset Filters"
    bl_description = "Reset filters to initial state (A and B only)"

    def invoke(self, context, event):
        return context.window_manager.invoke_confirm(self, event)

    def execute(self, context):
        scene = context.scene
        scene.custom_markers.clear()
        scene.marker_filters.clear()
        ensure_filters(scene)
        update_all_areas()
        self.report({'INFO'}, "Reset to initial state")
        return {'FINISHED'}

class MARKER_OT_add_filter(Operator):
    """フィルターを追加するオペレーター"""
    bl_idname = "marker.add_filter"
    bl_label = "Add Filter"
    bl_description = "Add a new marker filter"

    def execute(self, context):
        scene = context.scene
        ensure_filters(scene)

        existing_groups = [f.internal_group for f in scene.marker_filters if f.internal_group]
        # 次のアルファベット
        next_letter = None
        for i in range(26):
            letter = chr(ord('A') + i)
            if letter not in existing_groups:
                next_letter = letter
                break
        if next_letter is None:
            self.report({'WARNING'}, "Maximum 26 groups reached")
            return {'CANCELLED'}

        existing_colors = [(f.color[0], f.color[1], f.color[2]) for f in scene.marker_filters]

        colors = [
            (1.0, 0.3, 0.3, 1.0),  # 赤
            (0.3, 0.3, 1.0, 1.0),  # 青
            (0.3, 1.0, 0.3, 1.0),  # 緑
            (1.0, 1.0, 0.3, 1.0),  # 黄
            (1.0, 0.3, 1.0, 1.0),  # マゼンタ
            (0.3, 1.0, 1.0, 1.0),  # シアン
            (1.0, 0.6, 0.3, 1.0),  # オレンジ
            (0.6, 0.3, 1.0, 1.0),  # 紫
            (1.0, 0.3, 0.6, 1.0),  # ピンク
            (0.3, 1.0, 0.6, 1.0),  # ミント
            (0.7, 0.7, 0.7, 1.0),  # グレー
            (0.6, 0.4, 0.2, 1.0),  # 茶
        ]

        selected_color = None
        # 既存色とかぶらない色を選択（小数1桁で比較）
        def rgb3(x): return (round(x[0], 1), round(x[1], 1), round(x[2], 1))
        existing_rgb = {rgb3(c) for c in existing_colors}
        for c in colors:
            if rgb3(c) not in existing_rgb:
                selected_color = c
                break
        if selected_color is None:
            import random
            selected_color = (random.random(), random.random(), random.random(), 1.0)

        filter_item = scene.marker_filters.add()
        filter_item.internal_group = next_letter
        filter_item.filter_text = ""
        filter_item.color = selected_color

        self.report({'INFO'}, f"Added filter group {next_letter}")
        return {'FINISHED'}

class MARKER_OT_remove_filter(Operator):
    """フィルターを削除するオペレーター"""
    bl_idname = "marker.remove_filter"
    bl_label = "Remove Filter"
    bl_description = "Remove this filter and all its markers"

    filter_index: IntProperty()

    def invoke(self, context, event):
        return context.window_manager.invoke_confirm(self, event)

    @classmethod
    def poll(cls, context):
        return hasattr(context.scene, "marker_filters") and len(context.scene.marker_filters) > 2

    def execute(self, context):
        scene = context.scene
        ensure_filters(scene)

        if len(scene.marker_filters) > 2 and 0 <= self.filter_index < len(scene.marker_filters):
            filter_to_remove = scene.marker_filters[self.filter_index]
            group_to_remove = filter_to_remove.internal_group

            # 紐づくマーカー削除
            to_del = [i for i, m in enumerate(scene.custom_markers) if m.group == group_to_remove]
            for i in reversed(to_del):
                scene.custom_markers.remove(i)

            scene.marker_filters.remove(self.filter_index)
            update_all_areas()
            self.report({'INFO'}, f"Removed filter group {group_to_remove} and {len(to_del)} markers")
        return {'FINISHED'}

# ================================
# Draw Handler
# ================================

def draw_markers_in_editor(*_args):
    """各エディタにカスタムマーカーを描画（コンテキストは関数内で取得）"""
    context = bpy.context
    if not context or not context.scene:
        return

    scene = context.scene
    ensure_filters(scene)

    if not getattr(scene, "custom_markers", None):
        return

    space = getattr(context, "space_data", None)
    if not space or space.type not in {'DOPESHEET_EDITOR', 'GRAPH_EDITOR', 'NLA_EDITOR'}:
        return

    visible_markers = get_visible_markers(scene)
    if not visible_markers:
        return

    region = getattr(context, "region", None)
    if not region or not hasattr(region, 'view2d'):
        return
    view2d = region.view2d

    gpu.state.blend_set('ALPHA')
    gpu.state.line_width_set(1.0)

    shader = gpu.shader.from_builtin('UNIFORM_COLOR')

    for marker in visible_markers:
        try:
            x_pos, y_base = view2d.view_to_region(marker.frame, 0, clip=False)
        except Exception:
            continue
        if x_pos is None or x_pos < 0 or x_pos > region.width:
            continue

        # 縦ライン
        line_vertices = [(x_pos, 0), (x_pos, region.height)]
        line_batch = batch_for_shader(shader, 'LINES', {"pos": line_vertices})
        shader.bind()
        line_color = (*marker.color[:3], marker.color[3] * 0.5)
        shader.uniform_float("color", line_color)
        line_batch.draw(shader)

        # ▼三角
        y_bottom = 20
        tri = [(x_pos, y_bottom), (x_pos - 6, y_bottom + 10), (x_pos + 6, y_bottom + 10)]
        batch = batch_for_shader(shader, 'TRIS', {"pos": tri})
        shader.bind()
        shader.uniform_float("color", marker.color)
        batch.draw(shader)

        # 文字
        font_id = 0
        blf.position(font_id, x_pos + 8, y_bottom + 2, 0)
        blf.size(font_id, 11)
        blf.enable(font_id, blf.SHADOW)
        blf.shadow(font_id, 5, 0, 0, 0, 0.8)
        blf.shadow_offset(font_id, 1, -1)
        blf.color(font_id, 1, 1, 1, 1)
        blf.draw(font_id, marker.name)
        blf.disable(font_id, blf.SHADOW)

    gpu.state.blend_set('NONE')
    gpu.state.line_width_set(1.0)

# ================================
# Panels
# ================================

class CUSTOM_MARKER_PT_panel_base:
    """カスタムマーカーパネルの基底クラス"""
    bl_label = "Custom Markers"
    bl_region_type = 'UI'
    bl_category = "CM"

    def draw(self, context):
        layout = self.layout
        scene = context.scene
        ensure_filters(scene)

        # タイムラインコントロールを上部に配置
        timeline_row = layout.row(align=True)
        timeline_row.scale_y = 1.0
        
        # 再生/停止ボタン
        if context.screen.is_animation_playing:
            timeline_row.operator("screen.animation_play", text="", icon='PAUSE')
        else:
            timeline_row.operator("screen.animation_play", text="", icon='PLAY')
        
        # 現在のフレーム
        timeline_row.prop(scene, "frame_current", text="")
        
        timeline_row.separator()
        
        # 開始フレーム
        timeline_row.prop(scene, "frame_start", text="Start")
        
        # 終了フレーム
        timeline_row.prop(scene, "frame_end", text="End")

        # ━━━ フレーム範囲プリセットセクション ━━━
        box = layout.box()
        
        # 折りたたみ可能なヘッダー
        row = box.row(align=True)
        icon = 'DOWNARROW_HLT' if scene.show_frame_range_presets else 'RIGHTARROW'
        row.prop(scene, "show_frame_range_presets", text="", icon=icon, emboss=False)
        row.label(text="Frame Range Presets", icon='PREVIEW_RANGE')
        
        # 折りたたみ時は以降を表示しない
        if scene.show_frame_range_presets:
            # 選択中のキーフレームを取得（複数箇所で使用）
            selected_keyframes = get_selected_keyframes(context)
            
            # 選択中のキーフレーム操作
            if selected_keyframes:
                col = box.column(align=True)
                col.separator()
                
                row = col.row(align=True)
                row.label(text="Selected Keyframe:", icon='KEYFRAME')
                
                row = col.row(align=True)
                row.prop(scene, "selected_keyframe_frame", text="Frame")
                
                if len(selected_keyframes) > 1:
                    row = col.row()
                    row.scale_y = 0.8
                    row.label(text=f"({len(selected_keyframes)} keyframes selected)")
                
                col.separator()
            
            # 登録用の入力フィールド
            col = box.column(align=True)
            row = col.row(align=True)
            row.prop(scene, "frame_range_preset_start", text="Start")
            row.prop(scene, "frame_range_preset_end", text="End")
            
            row = col.row(align=True)
            row.scale_y = 1.2
            row.operator("frame_range.add_preset", text="Register Range", icon='ADD')
            
            # キーフレーム操作オプション
            if selected_keyframes:
                box.separator()
                col = box.column(align=True)
                col.label(text="Keyframe Options:", icon='KEYFRAME_HLT')
                col.prop(scene, "copy_keyframe_to_range_start_end", text="Copy to Start & End")
                col.prop(scene, "copy_keyframe_to_range_start_end_plus1", text="Copy to Start & End+1F")
                
                # 警告メッセージ（いずれかにチェックが入っている場合）
                if scene.copy_keyframe_to_range_start_end or scene.copy_keyframe_to_range_start_end_plus1:
                    col.separator()
                    warning_box = col.box()
                    warning_col = warning_box.column(align=True)
                    warning_col.scale_y = 0.9
                    warning_col.label(text="⚠ Copy Mode Active", icon='ERROR')
            
            # 登録済みプリセット一覧
            if len(scene.frame_range_presets) > 0:
                box.separator()
                col = box.column(align=True)
                
                for i, preset in enumerate(scene.frame_range_presets):
                    row = col.row(align=True)
                    
                    # 適用ボタン（範囲名を表示）
                    op = row.operator("frame_range.apply_preset", text=preset.get_name(), icon='PLAY')
                    op.preset_index = i
                    
                    # 削除ボタン
                    op = row.operator("frame_range.remove_preset", text="", icon='X')
                    op.preset_index = i
                
                # クリアボタン
                row = box.row()
                row.operator("frame_range.clear_all_presets", text="Clear All Presets", icon='TRASH')

        box = layout.box()

        # --- 標準＆カスタムマーカージャンプ（同列） ---
        row = box.row(align=True)
        row.alignment = 'CENTER'

        # 標準
        sub = row.row(align=True)
        btn = sub.row(); btn.scale_x = 0.3
        op = btn.operator("custom_marker.jump_to_standard", text="◀", icon='NONE'); op.direction = "PREV"
        sub.alignment = 'CENTER'; sub.label(text=" Default ")
        btn = sub.row(); btn.scale_x = 0.3
        op = btn.operator("custom_marker.jump_to_standard", text="▶", icon='NONE'); op.direction = "NEXT"

        # セパレーター
        row.separator(); row.separator(); row.separator()

        # カスタム
        sub = row.row(align=True)
        btn = sub.row(); btn.scale_x = 0.3
        op = btn.operator("custom_marker.jump_to_custom", text="◀", icon='NONE'); op.direction = "PREV"
        sub.alignment = 'CENTER'; sub.label(text=" Custom ")
        btn = sub.row(); btn.scale_x = 0.3
        op = btn.operator("custom_marker.jump_to_custom", text="▶", icon='NONE'); op.direction = "NEXT"

        # ── Start/End
        row = box.row()
        row.alignment = 'CENTER'
        grp = row.row(align=True)
        op = grp.operator("custom_marker.jump_to_end", text="", icon='REW'); op.direction = "FIRST"
        lbl = grp.row(); lbl.ui_units_x = 3.5; lbl.alignment = 'CENTER'
        lbl.label(text="Start / End", translate=False)
        op = grp.operator("custom_marker.jump_to_end", text="", icon='FF'); op.direction = "LAST"

        # フィルター
        box = layout.box()
        row = box.row()
        row.label(text="フィルター", icon='FILTER')
        row.operator("marker.add_filter", text=" Filter ", icon='ADD')
        row.operator("marker.clear_filters", text=" Clear Filter ", icon='PANEL_CLOSE')

        for i, filter_item in enumerate(scene.marker_filters):
            row = box.row(align=True)

            label_row = row.row(); label_row.scale_x = 0.3
            label_row.label(text=f"{filter_item.internal_group}:")

            op = row.operator("custom_marker.add", text="", icon='ADD')
            op.filter_index = i

            split = row.split(factor=0.55)
            split.prop(filter_item, "filter_text", text="")
            color_row = split.row(align=True)
            color_row.prop(filter_item, "color", text="")

            icon = 'HIDE_OFF' if filter_item.is_visible else 'HIDE_ON'
            color_row.prop(filter_item, "is_visible", text="", icon=icon, toggle=True)

            if len(scene.marker_filters) > 2:
                op = color_row.operator("marker.remove_filter", text="", icon='X')
                op.filter_index = i

        # マーカーリスト
        box = layout.box()
        row = box.row()
        row.label(text="Custom Markers", icon='MARKER')
        row.operator("custom_marker.clear_all", text="", icon='TRASH')

        visible_markers = get_visible_markers(scene)
        if visible_markers:
            for idx, marker in enumerate(scene.custom_markers):
                if marker in visible_markers:
                    row = box.row(align=True)
                    row.prop(marker, "color", text="")
                    sub_row = row.row(); sub_row.scale_x = 1.2; sub_row.label(text=f"{marker.name}")
                    sub_row = row.row(); sub_row.scale_x = 0.5; sub_row.label(text=f"F:{marker.frame}")
                    op = row.operator("custom_marker.jump_to", text="", icon='PLAY'); op.frame = marker.frame
                    op = row.operator("custom_marker.remove", text="", icon='X'); op.marker_index = idx
        else:
            box.label(text="No visible markers")

        layout.separator()
        col = layout.column(align=True); col.scale_y = 0.8
        col.label(text=f"Total: {len(scene.custom_markers)} markers", icon='INFO')
        col.label(text=f"Visible: {len(visible_markers)} markers")

class CUSTOM_MARKER_PT_panel_dopesheet(CUSTOM_MARKER_PT_panel_base, Panel):
    bl_idname = "CUSTOM_MARKER_PT_panel_dopesheet"
    bl_space_type = 'DOPESHEET_EDITOR'

class CUSTOM_MARKER_PT_panel_graph(CUSTOM_MARKER_PT_panel_base, Panel):
    bl_idname = "CUSTOM_MARKER_PT_panel_graph"
    bl_space_type = 'GRAPH_EDITOR'

class CUSTOM_MARKER_PT_panel_nla(CUSTOM_MARKER_PT_panel_base, Panel):
    bl_idname = "CUSTOM_MARKER_PT_panel_nla"
    bl_space_type = 'NLA_EDITOR'

# ================================
# Handlers (file load)
# ================================

@persistent
def load_post_ensure_filters(_dummy):
    # ファイル読込後に全シーンへ初期フィルターを保証
    for sc in bpy.data.scenes:
        ensure_filters(sc)
        # キーフレームコピーフラグを強制的にオフにする
        sc.copy_keyframe_to_range_start_end = False
        sc.copy_keyframe_to_range_start_end_plus1 = False

# ================================
# Register / Unregister
# ================================

_draw_handlers = []

classes = (
    CustomMarker,
    MarkerFilter,
    FrameRangePreset,
    FRAME_RANGE_OT_add,
    FRAME_RANGE_OT_apply,
    FRAME_RANGE_OT_remove,
    FRAME_RANGE_OT_clear_all,
    CUSTOM_MARKER_OT_add,
    CUSTOM_MARKER_OT_remove,
    CUSTOM_MARKER_OT_jump_to,
    CUSTOM_MARKER_OT_jump_to_end,
    CUSTOM_MARKER_OT_jump_to_standard,
    CUSTOM_MARKER_OT_jump_to_custom,
    CUSTOM_MARKER_OT_clear_all,
    MARKER_OT_add_filter,
    MARKER_OT_clear_filters,
    MARKER_OT_remove_filter,
    CUSTOM_MARKER_PT_panel_dopesheet,
    CUSTOM_MARKER_PT_panel_graph,
    CUSTOM_MARKER_PT_panel_nla,
)

def register():
    for cls in classes:
        bpy.utils.register_class(cls)

    bpy.types.Scene.custom_markers = CollectionProperty(type=CustomMarker)
    bpy.types.Scene.marker_filters = CollectionProperty(type=MarkerFilter)
    bpy.types.Scene.frame_range_presets = CollectionProperty(type=FrameRangePreset)
    
    # フレーム範囲プリセット用の入力フィールド
    bpy.types.Scene.frame_range_preset_start = IntProperty(
        name="Start Frame",
        description="Start frame for preset",
        default=1,
        min=0
    )
    bpy.types.Scene.frame_range_preset_end = IntProperty(
        name="End Frame",
        description="End frame for preset",
        default=250,
        min=1
    )
    
    # フレーム範囲プリセットセクションの表示/非表示
    bpy.types.Scene.show_frame_range_presets = BoolProperty(
        name="Show Frame Range Presets",
        description="Toggle Frame Range Presets section visibility",
        default=True
    )
    
    # キーフレーム操作オプション
    bpy.types.Scene.copy_keyframe_to_range_start_end = BoolProperty(
        name="Copy to Start & End",
        description="Copy selected keyframe to range start and end frames, then delete original (unless it overlaps)",
        default=False
    )
    bpy.types.Scene.copy_keyframe_to_range_start_end_plus1 = BoolProperty(
        name="Copy to Start & End+1F",
        description="Copy selected keyframe to range start and end+1 frames, then delete original (unless it overlaps)",
        default=False
    )
    
    # 選択中のキーフレームのフレーム番号（動的プロパティ）
    bpy.types.Scene.selected_keyframe_frame = IntProperty(
        name="Keyframe Frame",
        description="Frame number of selected keyframe",
        default=1,
        min=0,
        get=get_selected_keyframe_frame,
        set=set_selected_keyframe_frame
    )
    
    # フレーム入力用のプロパティを追加
    bpy.types.Scene.frame_input = IntProperty(
        name="Frame",
        description="Jump to specific frame",
        default=1,
        min=1,
        update=jump_to_input_frame
    )

    # ここで bpy.data.scenes を触らない（制限付きデータ対策）
    if load_post_ensure_filters not in bpy.app.handlers.load_post:
        bpy.app.handlers.load_post.append(load_post_ensure_filters)

    # Draw handlers（context を引数で捕まえない）
    global _draw_handlers
    _draw_handlers = []
    h = bpy.types.SpaceDopeSheetEditor.draw_handler_add(draw_markers_in_editor, (), 'WINDOW', 'POST_PIXEL')
    _draw_handlers.append(('DOPESHEET_EDITOR', h))
    h = bpy.types.SpaceGraphEditor.draw_handler_add(draw_markers_in_editor, (), 'WINDOW', 'POST_PIXEL')
    _draw_handlers.append(('GRAPH_EDITOR', h))
    h = bpy.types.SpaceNLA.draw_handler_add(draw_markers_in_editor, (), 'WINDOW', 'POST_PIXEL')
    _draw_handlers.append(('NLA_EDITOR', h))

def unregister():
    global _draw_handlers

    # Draw handlers remove
    for space_type, h in _draw_handlers:
        if space_type == 'DOPESHEET_EDITOR':
            bpy.types.SpaceDopeSheetEditor.draw_handler_remove(h, 'WINDOW')
        elif space_type == 'GRAPH_EDITOR':
            bpy.types.SpaceGraphEditor.draw_handler_remove(h, 'WINDOW')
        elif space_type == 'NLA_EDITOR':
            bpy.types.SpaceNLA.draw_handler_remove(h, 'WINDOW')
    _draw_handlers = []

    # load_post handler remove
    if load_post_ensure_filters in bpy.app.handlers.load_post:
        bpy.app.handlers.load_post.remove(load_post_ensure_filters)

    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)

    del bpy.types.Scene.custom_markers
    del bpy.types.Scene.marker_filters
    del bpy.types.Scene.frame_range_presets
    del bpy.types.Scene.frame_range_preset_start
    del bpy.types.Scene.frame_range_preset_end
    del bpy.types.Scene.show_frame_range_presets
    del bpy.types.Scene.copy_keyframe_to_range_start_end
    del bpy.types.Scene.copy_keyframe_to_range_start_end_plus1
    del bpy.types.Scene.selected_keyframe_frame
    del bpy.types.Scene.frame_input

if __name__ == "__main__":
    register()